<?php

namespace App\Http\Controllers\API\v1\Dashboard\Seller;

use App\Models\Banner;
use App\Helpers\ResponseError;
use Illuminate\Http\JsonResponse;
use App\Http\Resources\BannerResource;
use App\Http\Requests\DeleteAllRequest;
use App\Http\Requests\FilterParamsRequest;
use App\Services\BannerService\BannerService;
use Symfony\Component\HttpFoundation\Response;
use App\Http\Requests\Admin\Banner\StoreRequest;
use App\Http\Requests\Admin\Banner\UpdateRequest;
use App\Repositories\BannerRepository\BannerRepository;
use Illuminate\Http\Resources\Json\AnonymousResourceCollection;

class BannerController extends SellerBaseController
{

    public function __construct(
        protected BannerRepository $bannerRepository,
        protected BannerService $bannerService,
        protected Banner $model
    )
    {
        parent::__construct();
    }

    /**
     * Display a listing of the resource.
     */
    public function index(FilterParamsRequest $request): AnonymousResourceCollection
    {
        $banners = $this->bannerRepository->bannerPaginateSeller($request->perPage ?? 15, $this->shop->id);
        return BannerResource::collection($banners);
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param StoreRequest $request
     * @return JsonResponse
     */
    public function store(StoreRequest $request): JsonResponse
    {
        $collection = $request->validated();

        $result = $this->bannerService->create($collection, $this->shop->id);

        if ($result['status']) {
            return $this->successResponse(__('web.record_successfully_created'), BannerResource::make($result['data']));
        }
        return $this->errorResponse(
            $result['code'], $result['message'] ?? trans('errors.' . $result['code'], [], $this->language),
            Response::HTTP_BAD_REQUEST
        );
    }

    /**
     * Display the specified resource.
     *
     * @param int $id
     * @return JsonResponse
     */
    public function show(int $id): JsonResponse
    {
        $banner = $this->bannerRepository->bannerDetails($id);
        if ($banner) {
            return $this->successResponse(__('web.banner_found'), BannerResource::make($banner));
        }
        return $this->errorResponse(
            ResponseError::ERROR_404, trans('errors.' . ResponseError::ERROR_404, [], $this->language),
            Response::HTTP_NOT_FOUND
        );
    }

    /**
     * Update the specified resource in storage.
     *
     * @param UpdateRequest $request
     * @param int $id
     * @return JsonResponse
     */
    public function update(UpdateRequest $request, int $id): JsonResponse
    {
        $collection = $request->validated();
        $result = $this->bannerService->update($collection, $id, $this->shop->id);

        if ($result['status']) {
            return $this->successResponse(__('web.record_successfully_update'), BannerResource::make($result['data']));
        }
        return $this->errorResponse(
            $result['code'], $result['message'] ?? trans('errors.' . $result['code'], [], $this->language),
            Response::HTTP_BAD_REQUEST
        );
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param DeleteAllRequest $request
     * @return Response
     */
    public function destroy(DeleteAllRequest $request): Response
    {
        $collection = $request->validated();

        $result = $this->bannerService->delete($collection['ids']);

        if ($result['status']) {
            return $this->successResponse(__('web.record_has_been_successfully_delete'));
        }
        return $this->errorResponse(
            $result['code'], $result['message'] ?? trans('errors.' . $result['code'], [], $this->language),
            Response::HTTP_BAD_REQUEST
        );
    }

    /**
     * Change Active Status of Model.
     *
     * @param int $id
     * @return JsonResponse
     */
    public function setActiveBanner(int $id): JsonResponse
    {
        $banner = $this->model->find($id);
        if ($banner) {
            $banner->update(['active' => !$banner->active]);
            return $this->successResponse(__('web.record_has_been_successfully_updated'), BannerResource::make($banner));
        }
        return $this->errorResponse(
            ResponseError::ERROR_404, trans('errors.' . ResponseError::ERROR_404, [], $this->language),
            Response::HTTP_NOT_FOUND
        );
    }
}
